/*
 *  iot_sdio_ether.c - MT7682 SDIO interface driver
 *
 *  Copyright 2018 Sigma Star
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 *
 */

#include <linux/slab.h>
#include <linux/firmware.h>
#include <linux/delay.h>
#include <linux/mmc/card.h>
#include <linux/mmc/host.h>

#include <linux/mmc/sdio_func.h>
#include <linux/mmc/sdio_ids.h>


#include <linux/module.h>
#include <linux/bitops.h>

#include <linux/semaphore.h>

#include <linux/skbuff.h>
#include <linux/ip.h>

#include <linux/netdevice.h>
#include <linux/gpio.h>
#include <gpio.h>


#include <linux/platform_device.h>

#include "iot_api.h"
#include "iot_ether.h"
#include "iot_ctrl.h"
#include "mt7682_sdio.h"


#define __devinitconst 
#define __devinit 

/* Registers */
#define MAX_TX_COUNT_PER_WORKLOOP 16

#define VENDOR_AIROHA 0x037a
#define MT7682_DEV_ID 0x7682
#define MT7686_DEV_ID 0x7686

#define MT7682_MAX_TX_PKT_SIZE 1528
#define MT7682_TX_TIMEOUT_MS	10
#define MT7682_BUSY_DETECT_PIN PAD_PM_IRIN	//PAD_PM_GPIO8

extern void MHal_GPIO_Pull_Low(u8 u8IndexGPIO);
extern __init int mt7682_sdio_ops_init(void);

static const struct sdio_device_id mt7682_sdio_ids[] __devinitconst = {
	 {SDIO_DEVICE(SDIO_ANY_ID, MT7686_DEV_ID)},
	 {SDIO_DEVICE(SDIO_ANY_ID, MT7682_DEV_ID)},
	 {.driver_data = 0},

	{ /* end: all zeroes */ },
};



static void iot_eth_watchdog (PIOT_SDIO_HOST host)
{
	unsigned long time_to_chk = msecs_to_jiffies(MT7687_WATCHDOG_PERIOD);

	if (time_to_chk)
		mod_timer (&host->watchdog, jiffies + time_to_chk);
}

static void iot_eth_d2h_timer_handler (unsigned long data)
{
	PIOT_SDIO_HOST host = (PIOT_SDIO_HOST)data;
	set_bit (EVENT_D2H_INTR, &host->flags);	
	queue_work (host->iot_eth_work_queue, &host->iot_eth_work);

	mod_timer(&host->d2h_handler_timer, jiffies + msecs_to_jiffies(MT7687_WATCHDOG_PERIOD));
}

void iot_eth_update_iot_dev_info(void* para)
{


}

static int iot_eth_hard_xmit (PIOT_SDIO_HOST host, u32 isTcp, int num_pkt_to_send)
{
	int ret;
	struct sk_buff *tx_skb;	

	if(isTcp)
		tx_skb = iot_fill_skb_header(host->ndev, &host->tx_tcp_wait_q); 
	else
		tx_skb = iot_fill_skb_header(host->ndev, &host->tx_wait_q); 
	
	if (!tx_skb) {
	
		return 0;
	}
	
	
	if(tx_skb->len>(1528))	
		pr_err("len = %d\n",tx_skb->len);


	if(num_pkt_to_send==2)
	{
		
		struct sk_buff *tx_skb2;

		if(isTcp)
			tx_skb2 = iot_fill_skb_header(host->ndev, &host->tx_tcp_wait_q); 
		else
			tx_skb2 = iot_fill_skb_header(host->ndev, &host->tx_wait_q); 
		
		if (!tx_skb2) {
		
			return 0;
		}
		
		
		if(tx_skb2->len>(1528))	
			pr_err("len = %d\n",tx_skb2->len);

		iot_packet_send_2_pkt(tx_skb,tx_skb2 );
		dev_kfree_skb (tx_skb);	
		dev_kfree_skb (tx_skb2);

	}

	if(num_pkt_to_send==1)
	{

		ret = iot_packet_send(tx_skb);

		if(ret==0)
		{
			host->ndev->stats.tx_bytes += tx_skb->len;

			host->ndev->stats.tx_packets++;	
		}
		
		dev_kfree_skb (tx_skb);
	}


	return 1;
}


#if 0
int iot_eth_d2h_callback(iot_d2h_callback_event_type_t event_type, void *para)
{

	switch(event_type)
	{
		case IOT_D2H_EVENT_UPDATE_IOT_INFO:
//			set_bit(EVENT_D2H_RXQ1,&host->flags);

			break;

	}
	return 0;
}
#endif

static void iot_eth_work (struct work_struct *work)
{
#define OTHER_PKT 0
#define TCP_PKT 1

	int total_queue_len, total_eth1_queue_len = 0;
	PIOT_SDIO_HOST host = 
			container_of(work, IOT_SDIO_HOST, iot_eth_work);

	down (&host->lock);

#if 0	//TBD
	if (test_bit (EVENT_D2H_INTR, &host->flags)) {
		extern void mt7682_sdio_interrupt(struct sdio_func *func);
		
		pr_info("EVENT_D2H_INTR\r\n");
		mt7682_sdio_interrupt(host->func);

		clear_bit (EVENT_D2H_INTR, &host->flags);

	}


	if(test_bit(EVENT_IOT_INFO_UPDATED, &host->flags))
	{
		pr_info("EVENT_IOT_INFO_UPDATED\r\n");

	}
#endif

	if (test_bit (EVENT_TX, &host->flags)) {
		int n;
		int ret;
		int tx_count = 0;
		extern int sdio_func1_rd( unsigned int u4Register, void *pBuffer,	unsigned int Length);

		while((n=skb_queue_len(&host->tx_wait_q)))
		{
			if(gpio_get_value(host->busy_det_pin)==0)
			{
			
				if(n>1)
				{
					ret = iot_eth_hard_xmit (host, OTHER_PKT, 2 ); 
					if (!ret)
					{
						pr_warn("packet tx fail!\r\n");
					}
				}
				else if(n==1)
				{
					ret = iot_eth_hard_xmit (host, OTHER_PKT, 1 ); 
					if (!ret)
					{
						pr_warn("packet tx fail!\r\n");
					}
				}			
			}
			else
			{
				//pr_info("gpio_get_value(%d)=%d\r\n",host->busy_det_pin,gpio_get_value(host->busy_det_pin));				
				break;
			}

			tx_count++;
			if(tx_count==MAX_TX_COUNT_PER_WORKLOOP)
				break;
				
		}

#if 0

		if(whisr & 0x10000/*SDIO_SWINT_MB0_BROMRECV*/)
		{
			volatile uint32_t  mail_value = 0;
			extern bool h2d_receive_mailbox( uint32_t *cmd);
			sdio_claim_host(host->func);		
			h2d_receive_mailbox(&mail_value);
			sdio_release_host(host->func);			
			pr_info("###h2d_receive_mailbox: value = %d\r\n",mail_value);

			if(mail_value == 0x84 )
			{
				pr_info("netif_carrier_on");
				netif_carrier_on(host->ndev);
			}
			else
			{
				pr_info("netif_carrier_off");
				netif_carrier_off(host->ndev);
			}
		}
#endif		
#if 0
		tx_count = 0;
		while ((n = skb_queue_len(&eth_priv_data->tx_tcp_wait_q))) 
		{
		
			ret = iot_eth1_hard_xmit (eth_priv_data, TCP_PKT); 
			
			if (!ret)
			{
				break;	
			}
			tx_count++;
			if(tx_count==MAX_TX_COUNT_PER_WORKLOOP)
				break;
			
		}

		tx_count = 0;
		while ((n = skb_queue_len(&eth_priv_data->tx_wait_q))) {
			
			ret = iot_eth1_hard_xmit (eth_priv_data, OTHER_PKT ); 
			if (!ret)
			{
				break;	
			}
			tx_count++;
			if(tx_count==MAX_TX_COUNT_PER_WORKLOOP)
				break;
		}
#endif
		n = skb_queue_len(&host->tx_wait_q);
		if(n==0)
			clear_bit (EVENT_TX, &host->flags);

		total_queue_len = (skb_queue_len(&host->tx_wait_q)+skb_queue_len(&host->tx_tcp_wait_q)) ;
		if (netif_queue_stopped (host->ndev) && 
		    ((total_queue_len) < TX_QUEUE_LOW_THRESHOLD))
		{
			netif_wake_queue (host->ndev);
		}

		if(total_queue_len+total_eth1_queue_len)
		{
			set_bit (EVENT_TX, &host->flags);
			queue_work (host->iot_eth_work_queue, &host->iot_eth_work);
		}		
	}

	if (test_bit (EVENT_SET_WIFI, &host->flags)) {
		printk(KERN_INFO "EVENT_SET_WIFI\n");

		if(host->pWifiSetting){
			iot_set_wifi(host->pWifiSetting);		
			host->pWifiSetting = 0;
		}

		clear_bit (EVENT_SET_WIFI, &host->flags);

	}

	if (test_bit (EVENT_WATCHDOG, &host->flags)) {
		iot_eth_watchdog (host);

		clear_bit (EVENT_WATCHDOG, &host->flags);
	}


	up (&host->lock);	

}



static void iot_eth_set_mac_addr (struct net_device *ndev)
{
	int ret;

	char* mac = ndev->dev_addr;

	ret = iot_get_mac(0, mac );
	printk("mtk_iot_set_mac_addr: ret=%d %02x:%02x:%02x:%02x:%02x:%02x\r\n",ret,mac[0], mac[1], mac[2], mac[3], mac[4], mac[5] );

	return;
}



int iot_eth_open(struct net_device *ndev)
{

	PIOT_SDIO_HOST host = (PIOT_SDIO_HOST)netdev_priv(ndev);
	
	netif_carrier_off (ndev);

	host->seq_num = 0x1f;

	netif_start_queue (ndev);

	netif_carrier_on (ndev);

	return 0;
}


int	iot_eth_close(struct net_device *dev)
{
	PIOT_SDIO_HOST host = (PIOT_SDIO_HOST)netdev_priv(dev);

	sdio_claim_host(host->func);
	sdio_release_irq(host->func);
	sdio_release_host(host->func);

	iot_close(host->func);
	
	pr_info("%s is closed.\r\n",dev->name);

	return 0;
}


netdev_tx_t	iot_eth_start_xmit (struct sk_buff *skb,
					   struct net_device *ndev)
{

	int n;
	PIOT_SDIO_HOST host = (PIOT_SDIO_HOST)netdev_priv(ndev);

#ifdef TX_PKT_PERIOD_SEND
	if(ip_hdr(skb)->protocol == IPPROTO_TCP)
	{
		skb_queue_tail(&host->tx_tcp_wait_q, skb);

		if ((n=skb_queue_len (&host->tx_tcp_wait_q)) > (TX_QUEUE_HIGH_THRESHOLD/2)) {
			if (netif_msg_tx_queued (host))
				printk ("%s: Too much TX(TCP) packets in queue %d\n"
					, __FUNCTION__
					, skb_queue_len (&host->tx_tcp_wait_q));
			netif_stop_queue (ndev);
			return NETDEV_TX_OK;
		}	
	}
	else
#endif	

	skb_queue_tail (&host->tx_wait_q, skb);

	if ((n=skb_queue_len (&host->tx_wait_q)) > TX_QUEUE_HIGH_THRESHOLD) {
		if (netif_msg_tx_queued (host))
			printk ("%s: Too much TX(UDP) packets in queue %d\n"
				, __FUNCTION__
				, skb_queue_len (&host->tx_wait_q));
		netif_stop_queue (ndev);
	}
		
	set_bit (EVENT_TX, &host->flags);
	queue_work (host->iot_eth_work_queue, &host->iot_eth_work);
	
	return NETDEV_TX_OK;

}					   


static int iot_eth_set_mac_address (struct net_device *ndev, void *p)
{
	struct sockaddr *addr = p;

	if (!is_valid_ether_addr(addr->sa_data))
		return -EADDRNOTAVAIL;

	memcpy(ndev->dev_addr, addr->sa_data, ndev->addr_len);
	iot_eth_set_mac_addr (ndev);


	return 0;
}


int iot_eth_ioctl(struct net_device *dev,
				        struct ifreq *ifr, int cmd)
{
	extern long iot_priv_ioctl(struct net_device *dev, unsigned int cmd, void* arg);
	return iot_priv_ioctl(dev, cmd, ifr->ifr_data);

}
						

struct net_device_stats* iot_eth_get_stats(struct net_device *dev)
{
	BUG_ON(!dev);

	return &dev->stats;
}


static const struct net_device_ops iot_eth_netdev_ops = {
	.ndo_open		= iot_eth_open,
	.ndo_stop		= iot_eth_close,
	.ndo_start_xmit		= iot_eth_start_xmit,
	.ndo_get_stats		= iot_eth_get_stats,
	.ndo_do_ioctl		= iot_eth_ioctl,
	.ndo_set_mac_address	= iot_eth_set_mac_address,
};


static int __devinit mt7682_sdio_probe(struct sdio_func *func,
			 const struct sdio_device_id *id)
{
	int ret;
	char devname[] = "iot";
	struct net_device *ndev;
	PIOT_SDIO_HOST host;
	extern iot_dev_info_t gDevinfo;

	ndev = alloc_etherdev (sizeof (IOT_SDIO_HOST));
	if (!ndev) {
		printk(KERN_ERR
			   "iot_eth_probe: Could not allocate ethernet device\n");
		return -ENOMEM;
	}

	ndev->netdev_ops	= &iot_eth_netdev_ops;
	ndev->ethtool_ops	= &iot_ethtool_ops;

	ndev->features &= ~NETIF_F_HW_CSUM;
	ndev->hard_header_len += (TX_OVERHEAD+4);

	snprintf(ndev->name, sizeof(ndev->name), "%s%u", devname, 0);
	
	ret = register_netdev(ndev);
	if (ret)
	{	
		printk(KERN_INFO "Netdev register fail. (error = %d)", ret);
		goto exit;
	}

	host = (PIOT_SDIO_HOST)netdev_priv(ndev);
	host->ndev = ndev;

	INIT_WORK(&host->iot_eth_work, iot_eth_work);

	setup_timer(&host->d2h_handler_timer, iot_eth_d2h_timer_handler,
		    (unsigned long) host);

	host->board_id = id->driver_data;

	
	host->iot_eth_work_queue = create_singlethread_workqueue ("iot_eth_work");
	
	sema_init (&host->lock,1);

	host->func = func;

	host->device = &func->dev;
	host->busy_det_pin = MT7682_BUSY_DETECT_PIN;

	skb_queue_head_init(&host->tx_tcp_wait_q);
	skb_queue_head_init(&host->tx_wait_q);
	sdio_set_drvdata(func, host);

//	mt7682_sdio_ops_init();
	
	ret = iot_init(func );
	if(ret<0)
	{
		printk(KERN_INFO "iot init failed. (error = %d)",ret);	
	
		goto exit_unreg_netdev;
	}

	ret = iot_get_info( 0, &gDevinfo);
	pr_info("IoT Link Status = x%x\r\n",(int)gDevinfo.status);

	iot_eth_set_mac_addr (ndev);


	if(gDevinfo.status==0x84)
	{
#ifdef CONFIG_IP_PNP
		extern int __init iot_ip_auto_config_setup(char *addrs);
		char ip_cfg_str[100];
		__be32 ipaddr, gateway_addr, netmask;

		ipaddr = gDevinfo.ipaddr;
		gateway_addr  = gDevinfo.gateway_addr;
		netmask  = gDevinfo.netmask;
		
		sprintf(ip_cfg_str,"%d.%d.%d.%d::%d.%d.%d.%d:%d.%d.%d.%d:none:eth0:on",ipaddr&0xff, (ipaddr>>8)&0xff,(ipaddr>>16)&0xff, ipaddr>>24,
								gateway_addr&0xff, (gateway_addr>>8)&0xff,(gateway_addr>>16)&0xff, gateway_addr>>24,
								netmask&0xff, (netmask>>8)&0xff,(netmask>>16)&0xff, netmask>>24);
		printk("iot_eth:%s\r\n",ip_cfg_str);
		iot_ip_auto_config_setup(ip_cfg_str);
#endif		

	}

	ret = gpio_request(host->busy_det_pin, "busy detect pin");
	if (ret) {
		pr_err("Failed requesting gpio_ro %d\n", host->busy_det_pin);
		return -1;
	}

	gpio_direction_input(host->busy_det_pin);

	
	return 0;
	
exit_unreg_netdev:
	unregister_netdev(ndev);
	destroy_workqueue(host->iot_eth_work_queue);	
exit:

	free_netdev(ndev);

	return ret;
}

static void mt7682_sdio_remove(struct sdio_func *func)
{
	pr_info("%s: todo\r\n",__func__);
	gpio_free(MT7682_BUSY_DETECT_PIN);
}


static struct sdio_driver mt7682_sdio_driver = {
	.name = "mt7682sdio",
	.id_table = mt7682_sdio_ids,
	.probe = mt7682_sdio_probe,
	.remove = mt7682_sdio_remove,
};

/*******************************************************************/
/* Module functions                                                */
/*******************************************************************/

 
static int mt7682_sdio_module_init(void)
{
	int ret = 0;

	printk(KERN_INFO "mt7682 sdio module is initializing.\n");

	ret = sdio_register_driver(&mt7682_sdio_driver);

	return ret;
}

static void __exit mt7682_sdio_module_exit(void)
{
	sdio_unregister_driver(&mt7682_sdio_driver);
}

module_init(mt7682_sdio_module_init);
module_exit(mt7682_sdio_module_exit);
